GQuark  bd_nvme_error_quark (void) {
        return g_quark_from_static_string ("g-bd-nvme-error-quark");
}

/**
 * BDNVMEControllerInfo:
 * @pci_vendor_id: The PCI Vendor ID.
 * @pci_subsys_vendor_id: The PCI Subsystem Vendor ID.
 * @ctrl_id: Controller ID, the NVM subsystem unique controller identifier associated with the controller.
 * @fguid: FRU GUID, a 128-bit value that is globally unique for a given Field Replaceable Unit.
 * @model_number: The model number.
 * @serial_number: The serial number.
 * @firmware_ver: The currently active firmware revision.
 * @nvme_ver: The NVM Express base specification that the controller implementation supports or %NULL when not reported by the device.
 * @features: features and capabilities present for this controller, see #BDNVMEControllerFeature.
 * @controller_type: The controller type.
 * @selftest_ext_time: Extended Device Self-test Time, if #BD_NVME_CTRL_FEAT_SELFTEST is supported then this field
 *                     indicates the nominal amount of time in one minute units that the controller takes
 *                     to complete an extended device self-test operation when in power state 0.
 * @hmb_pref_size: Host Memory Buffer Preferred Size indicates the preferred size that the host
 *                 is requested to allocate for the Host Memory Buffer feature in bytes.
 * @hmb_min_size: Host Memory Buffer Minimum Size indicates the minimum size that the host
 *                is requested to allocate for the Host Memory Buffer feature in bytes.
 * @size_total: Total NVM Capacity in the NVM subsystem in bytes.
 * @size_unalloc: Unallocated NVM Capacity in the NVM subsystem in bytes.
 * @num_namespaces: Maximum Number of Allowed Namespaces supported by the NVM subsystem.
 * @subsysnqn: NVM Subsystem NVMe Qualified Name, UTF-8 null terminated string.
 */
/**
 * bd_nvme_controller_info_free: (skip)
 * @info: (nullable): %BDNVMEControllerInfo to free
 *
 * Frees @info.
 */
void  bd_nvme_controller_info_free (BDNVMEControllerInfo *info) {
        if (info == NULL)
        return;

    g_free (info->fguid);
    g_free (info->subsysnqn);
    g_free (info->model_number);
    g_free (info->serial_number);
    g_free (info->firmware_ver);
    g_free (info->nvme_ver);
    g_free (info);
}

/**
 * bd_nvme_controller_info_copy: (skip)
 * @info: (nullable): %BDNVMEControllerInfo to copy
 *
 * Creates a new copy of @info.
 */
BDNVMEControllerInfo * bd_nvme_controller_info_copy (BDNVMEControllerInfo *info) {
        BDNVMEControllerInfo *new_info;

    if (info == NULL)
        return NULL;

    new_info = g_new0 (BDNVMEControllerInfo, 1);
    memcpy (new_info, info, sizeof (BDNVMEControllerInfo));
    new_info->fguid = g_strdup (info->fguid);
    new_info->subsysnqn = g_strdup (info->subsysnqn);
    new_info->model_number = g_strdup (info->model_number);
    new_info->serial_number = g_strdup (info->serial_number);
    new_info->firmware_ver = g_strdup (info->firmware_ver);
    new_info->nvme_ver = g_strdup (info->nvme_ver);

    return new_info;
}

GType  bd_nvme_controller_info_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMEControllerInfo",
                                             (GBoxedCopyFunc) bd_nvme_controller_info_copy,
                                             (GBoxedFreeFunc) bd_nvme_controller_info_free);
    }
    return type;
}

/**
 * BDNVMELBAFormat:
 * Namespace LBA Format Data Structure.
 * @data_size: LBA data size (i.e. a sector size) in bytes.
 * @metadata_size: metadata size in bytes or `0` in case of no metadata support.
 * @relative_performance: Relative Performance index, see #BDNVMELBAFormatRelativePerformance.
 */
/**
 * bd_nvme_lba_format_free: (skip)
 * @fmt: (nullable): %BDNVMELBAFormat to free
 *
 * Frees @fmt.
 */
void  bd_nvme_lba_format_free (BDNVMELBAFormat *fmt) {
        g_free (fmt);
}

/**
 * bd_nvme_lba_format_copy: (skip)
 * @fmt: (nullable): %BDNVMELBAFormat to copy
 *
 * Creates a new copy of @fmt.
 */
BDNVMELBAFormat * bd_nvme_lba_format_copy (BDNVMELBAFormat *fmt) {
        BDNVMELBAFormat *new_fmt;

    if (fmt == NULL)
        return NULL;

    new_fmt = g_new0 (BDNVMELBAFormat, 1);
    new_fmt->data_size = fmt->data_size;
    new_fmt->metadata_size = fmt->metadata_size;
    new_fmt->relative_performance = fmt->relative_performance;

    return new_fmt;
}

GType  bd_nvme_lba_format_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMELBAFormat",
                                             (GBoxedCopyFunc) bd_nvme_lba_format_copy,
                                             (GBoxedFreeFunc) bd_nvme_lba_format_free);
    }
    return type;
}

/**
 * BDNVMENamespaceInfo:
 * @nsid: The Namespace Identifier (NSID).
 * @eui64: IEEE Extended Unique Identifier: a 64-bit IEEE Extended Unique Identifier (EUI-64)
 *         that is globally unique and assigned to the namespace when the namespace is created.
 *         Remains fixed throughout the life of the namespace and is preserved across namespace
 *         and controller operations.
 * @nguid: Namespace Globally Unique Identifier: a 128-bit value that is globally unique and
 *         assigned to the namespace when the namespace is created. Remains fixed throughout
 *         the life of the namespace and is preserved across namespace and controller operations.
 * @uuid: Namespace 128-bit Universally Unique Identifier (UUID) as specified in RFC 4122.
 * @nsize: Namespace Size: total size of the namespace in logical blocks. The number of logical blocks
 *         is based on the formatted LBA size (see @current_lba_format).
 * @ncap: Namespace Capacity: maximum number of logical blocks that may be allocated in the namespace
 *        at any point in time. The number of logical blocks is based on the formatted LBA size (see @current_lba_format).
 * @nuse: Namespace Utilization: current number of logical blocks allocated in the namespace.
 *        This field is smaller than or equal to the Namespace Capacity. The number of logical
 *        blocks is based on the formatted LBA size (see @current_lba_format).
 * @features: features and capabilities present for this namespace, see #BDNVMENamespaceFeature.
 * @format_progress_remaining: The percentage value remaining of a format operation in progress.
 * @write_protected: %TRUE if the namespace is currently write protected and all write access to the namespace shall fail.
 * @lba_formats: (array zero-terminated=1): A list of supported LBA Formats.
 * @current_lba_format: A LBA Format currently used for the namespace. Contains zeroes in case of
 *                      an invalid or no supported LBA Format reported.
 */
/**
 * bd_nvme_namespace_info_free: (skip)
 * @info: (nullable): %BDNVMENamespaceInfo to free
 *
 * Frees @info.
 */
void  bd_nvme_namespace_info_free (BDNVMENamespaceInfo *info) {
        BDNVMELBAFormat **lba_formats;

    if (info == NULL)
        return;

    g_free (info->eui64);
    g_free (info->uuid);
    g_free (info->nguid);

    for (lba_formats = info->lba_formats; lba_formats && *lba_formats; lba_formats++)
        bd_nvme_lba_format_free (*lba_formats);
    g_free (info->lba_formats);
    g_free (info);
}

/**
 * bd_nvme_namespace_info_copy: (skip)
 * @info: (nullable): %BDNVMENamespaceInfo to copy
 *
 * Creates a new copy of @info.
 */
BDNVMENamespaceInfo * bd_nvme_namespace_info_copy (BDNVMENamespaceInfo *info) {
        BDNVMENamespaceInfo *new_info;
    BDNVMELBAFormat **lba_formats;
    GPtrArray *ptr_array;

    if (info == NULL)
        return NULL;

    new_info = g_new0 (BDNVMENamespaceInfo, 1);
    memcpy (new_info, info, sizeof (BDNVMENamespaceInfo));
    new_info->eui64 = g_strdup (info->eui64);
    new_info->uuid = g_strdup (info->uuid);
    new_info->nguid = g_strdup (info->nguid);

    ptr_array = g_ptr_array_new ();
    for (lba_formats = info->lba_formats; lba_formats && *lba_formats; lba_formats++)
        g_ptr_array_add (ptr_array, bd_nvme_lba_format_copy (*lba_formats));
    g_ptr_array_add (ptr_array, NULL);
    new_info->lba_formats = (BDNVMELBAFormat **) g_ptr_array_free (ptr_array, FALSE);

    return new_info;
}

GType  bd_nvme_namespace_info_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMENamespaceInfo",
                                             (GBoxedCopyFunc) bd_nvme_namespace_info_copy,
                                             (GBoxedFreeFunc) bd_nvme_namespace_info_free);
    }
    return type;
}

/**
 * BDNVMESmartLog:
 * @critical_warning: critical warnings for the state of the controller, see #BDNVMESmartCriticalWarning.
 * @avail_spare: Available Spare: a normalized percentage (0% to 100%) of the remaining spare capacity available.
 * @spare_thresh: Available Spare Threshold: a normalized percentage (0% to 100%) of the available spare threshold.
 * @percent_used: Percentage Used: a vendor specific estimate of the percentage drive life used based on the
 *                actual usage and the manufacturer's prediction. A value of 100 indicates that the estimated
 *                endurance has been consumed, but may not indicate an NVM subsystem failure.
 *                The value is allowed to exceed 100.
 * @total_data_read: An estimated calculation of total data read in bytes based on calculation of data
 *                   units read from the host. A value of 0 indicates that the number of Data Units Read
 *                   is not reported.
 * @total_data_written: An estimated calculation of total data written in bytes based on calculation
 *                      of data units written by the host. A value of 0 indicates that the number
 *                      of Data Units Written is not reported.
 * @ctrl_busy_time: Amount of time the controller is busy with I/O commands, reported in minutes.
 * @power_cycles: The number of power cycles.
 * @power_on_hours: The number of power-on hours, excluding a non-operational power state.
 * @unsafe_shutdowns: The number of unsafe shutdowns as a result of a Shutdown Notification not received prior to loss of power.
 * @media_errors: Media and Data Integrity Errors: the number of occurrences where the controller detected
 *                an unrecovered data integrity error (e.g. uncorrectable ECC, CRC checksum failure, or LBA tag mismatch).
 * @num_err_log_entries: Number of Error Information Log Entries: the number of Error Information log
 *                       entries over the life of the controller.
 * @temperature: Composite Temperature: temperature in Kelvins that represents the current composite
 *               temperature of the controller and associated namespaces or 0 when not applicable.
 * @temp_sensors: Temperature Sensor 1-8: array of the current temperature reported by temperature sensors
 *                1-8 in Kelvins or 0 when the particular sensor is not available.
 * @wctemp: Warning Composite Temperature Threshold (WCTEMP): indicates the minimum Composite Temperature (@temperature)
 *          value that indicates an overheating condition during which controller operation continues.
 *          A value of 0 indicates that no warning temperature threshold value is reported by the controller.
 * @cctemp: Critical Composite Temperature Threshold (CCTEMP): indicates the minimum Composite Temperature (@temperature)
 *          value that indicates a critical overheating condition (e.g., may prevent continued normal operation,
 *          possibility of data loss, automatic device shutdown, extreme performance throttling, or permanent damage).
 *          A value of 0 indicates that no critical temperature threshold value is reported by the controller.
 * @warning_temp_time: Warning Composite Temperature Time: the amount of time in minutes that the Composite Temperature (@temperature)
 *                     is greater than or equal to the Warning Composite Temperature Threshold (@wctemp) and less than the
 *                     Critical Composite Temperature Threshold (@cctemp).
 * @critical_temp_time: Critical Composite Temperature Time: the amount of time in minutes that the Composite Temperature (@temperature)
 *                      is greater than or equal to the Critical Composite Temperature Threshold (@cctemp).
 */
/**
 * bd_nvme_smart_log_free: (skip)
 * @log: (nullable): %BDNVMESmartLog to free
 *
 * Frees @log.
 */
void  bd_nvme_smart_log_free (BDNVMESmartLog *log) {
        g_free (log);
}

/**
 * bd_nvme_smart_log_copy: (skip)
 * @log: (nullable): %BDNVMESmartLog to copy
 *
 * Creates a new copy of @log.
 */
BDNVMESmartLog * bd_nvme_smart_log_copy (BDNVMESmartLog *log) {
        BDNVMESmartLog *new_log;

    if (log == NULL)
        return NULL;

    new_log = g_new0 (BDNVMESmartLog, 1);
    memcpy (new_log, log, sizeof (BDNVMESmartLog));

    return new_log;
}

GType  bd_nvme_smart_log_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMESmartLog",
                                             (GBoxedCopyFunc) bd_nvme_smart_log_copy,
                                             (GBoxedFreeFunc) bd_nvme_smart_log_free);
    }
    return type;
}

/**
 * BDNVMEErrorLogEntry:
 * @error_count: internal error counter, a unique identifier for the error.
 * @command_id: the Command Identifier of the command that the error is associated with or `0xffff` if the error is not specific to a particular command.
 * @command_specific: Command Specific Information specific to @command_id.
 * @command_status: the Status code for the command that completed.
 * @command_error: translated command error in the BD_NVME_ERROR domain or %NULL in case @command_status indicates success.
 * @lba: the first LBA that experienced the error condition.
 * @nsid: the NSID of the namespace that the error is associated with.
 * @transport_type: type of the transport associated with the error.
 */
/**
 * bd_nvme_error_log_entry_free: (skip)
 * @entry: (nullable): %BDNVMEErrorLogEntry to free
 *
 * Frees @entry.
 */
void  bd_nvme_error_log_entry_free (BDNVMEErrorLogEntry *entry) {
        if (entry == NULL)
        return;

    if (entry->command_error)
        g_error_free (entry->command_error);
    g_free (entry);
}

/**
 * bd_nvme_error_log_entry_copy: (skip)
 * @entry: (nullable): %BDNVMEErrorLogEntry to copy
 *
 * Creates a new copy of @entry.
 */
BDNVMEErrorLogEntry * bd_nvme_error_log_entry_copy (BDNVMEErrorLogEntry *entry) {
        BDNVMEErrorLogEntry *new_entry;

    if (entry == NULL)
        return NULL;

    new_entry = g_new0 (BDNVMEErrorLogEntry, 1);
    memcpy (new_entry, entry, sizeof (BDNVMEErrorLogEntry));
    if (entry->command_error)
        new_entry->command_error = g_error_copy (entry->command_error);

    return new_entry;
}

GType  bd_nvme_error_log_entry_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMEErrorLogEntry",
                                             (GBoxedCopyFunc) bd_nvme_error_log_entry_copy,
                                             (GBoxedFreeFunc) bd_nvme_error_log_entry_free);
    }
    return type;
}

/**
 * BDNVMESelfTestLogEntry:
 * @result: Result of the device self-test operation.
 * @action: The Self-test Code value (action) that was specified in the Device Self-test command that started this device self-test operation.
 * @segment: Segment number where the first self-test failure occurred. Valid only when @result is set to #BD_NVME_SELF_TEST_RESULT_KNOWN_SEG_FAIL.
 * @power_on_hours: Number of power-on hours at the time the device self-test operation was completed or aborted. Does not include time that the controller was powered and in a low power state condition.
 * @nsid: Namespace ID that the Failing LBA occurred on.
 * @failing_lba: LBA of the logical block that caused the test to fail. If the device encountered more than one failed logical block during the test, then this field only indicates one of those failed logical blocks.
 * @status_code_error: Translated NVMe Command Status Code representing additional information related to errors or conditions.
 */
/**
 * bd_nvme_self_test_log_entry_free: (skip)
 * @entry: (nullable): %BDNVMESelfTestLogEntry to free
 *
 * Frees @entry.
 */
void  bd_nvme_self_test_log_entry_free (BDNVMESelfTestLogEntry *entry) {
        if (entry == NULL)
        return;

    if (entry->status_code_error)
        g_error_free (entry->status_code_error);
    g_free (entry);
}

/**
 * bd_nvme_self_test_log_entry_copy: (skip)
 * @entry: (nullable): %BDNVMESelfTestLogEntry to copy
 *
 * Creates a new copy of @entry.
 */
BDNVMESelfTestLogEntry * bd_nvme_self_test_log_entry_copy (BDNVMESelfTestLogEntry *entry) {
        BDNVMESelfTestLogEntry *new_entry;

    if (entry == NULL)
        return NULL;

    new_entry = g_new0 (BDNVMESelfTestLogEntry, 1);
    memcpy (new_entry, entry, sizeof (BDNVMESelfTestLogEntry));
    if (entry->status_code_error)
        new_entry->status_code_error = g_error_copy (entry->status_code_error);

    return new_entry;
}

GType  bd_nvme_self_test_log_entry_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMESelfTestLogEntry",
                                             (GBoxedCopyFunc) bd_nvme_self_test_log_entry_copy,
                                             (GBoxedFreeFunc) bd_nvme_self_test_log_entry_free);
    }
    return type;
}

/**
 * BDNVMESelfTestLog:
 * @current_operation: Current running device self-test operation. There's no corresponding record in @entries for a device self-test operation that is in progress.
 * @current_operation_completion: Percentage of the currently running device self-test operation. Only valid when @current_operation is other than #BD_NVME_SELF_TEST_ACTION_NOT_RUNNING.
 * @entries: (array zero-terminated=1): Self-test log entries for the last 20 operations, sorted from newest (first element) to oldest.
 */
/**
 * bd_nvme_self_test_log_free: (skip)
 * @log: (nullable): %BDNVMESelfTestLog to free
 *
 * Frees @log.
 */
void  bd_nvme_self_test_log_free (BDNVMESelfTestLog *log) {
        BDNVMESelfTestLogEntry **entries;

    if (log == NULL)
        return;

    for (entries = log->entries; entries && *entries; entries++)
        bd_nvme_self_test_log_entry_free (*entries);
    g_free (log->entries);
    g_free (log);
}

/**
 * bd_nvme_self_test_log_copy: (skip)
 * @log: (nullable): %BDNVMESelfTestLog to copy
 *
 * Creates a new copy of @log.
 */
BDNVMESelfTestLog * bd_nvme_self_test_log_copy (BDNVMESelfTestLog *log) {
        BDNVMESelfTestLog *new_log;
    BDNVMESelfTestLogEntry **entries;
    GPtrArray *ptr_array;

    if (log == NULL)
        return NULL;

    new_log = g_new0 (BDNVMESelfTestLog, 1);
    memcpy (new_log, log, sizeof (BDNVMESelfTestLog));

    ptr_array = g_ptr_array_new ();
    for (entries = log->entries; entries && *entries; entries++)
        g_ptr_array_add (ptr_array, bd_nvme_self_test_log_entry_copy (*entries));
    g_ptr_array_add (ptr_array, NULL);
    new_log->entries = (BDNVMESelfTestLogEntry **) g_ptr_array_free (ptr_array, FALSE);

    return new_log;
}

GType  bd_nvme_self_test_log_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMESelfTestLog",
                                             (GBoxedCopyFunc) bd_nvme_self_test_log_copy,
                                             (GBoxedFreeFunc) bd_nvme_self_test_log_free);
    }
    return type;
}

/**
 * BDNVMESanitizeLog:
 * @sanitize_progress: The percentage complete of the sanitize operation.
 * @sanitize_status: The status of the most recent sanitize operation.
 * @global_data_erased: Indicates that no user data has been written either since the drive was manufactured and
 *                      has never been sanitized or since the most recent successful sanitize operation.
 * @overwrite_passes: Number of completed passes if the most recent sanitize operation was an Overwrite.
 * @time_for_overwrite: Estimated time in seconds needed to complete an Overwrite sanitize operation with 16 passes in the background.
 *                      A value of -1 means that no time estimate is reported. A value of 0 means that the operation is expected
 *                      to be completed in the background when the Sanitize command is completed.
 * @time_for_block_erase: Estimated time in seconds needed to complete a Block Erase sanitize operation in the background.
 *                        A value of -1 means that no time estimate is reported. A value of 0 means that the operation is expected
 *                        to be completed in the background when the Sanitize command is completed.
 * @time_for_crypto_erase: Estimated time in seconds needed to complete a Crypto Erase sanitize operation in the background.
 *                         A value of -1 means that no time estimate is reported. A value of 0 means that the operation is expected
 *                         to be completed in the background when the Sanitize command is completed.
 * @time_for_overwrite_nd: Estimated time in seconds needed to complete an Overwrite sanitize operation and the associated
 *                         additional media modification in the background when the No-Deallocate After Sanitize or
 *                         the No-Deallocate Modifies Media After Sanitize features have been requested.
 * @time_for_block_erase_nd: Estimated time in seconds needed to complete a Block Erase sanitize operation and the associated
 *                           additional media modification in the background when the No-Deallocate After Sanitize or
 *                           the No-Deallocate Modifies Media After Sanitize features have been requested.
 * @time_for_crypto_erase_nd: Estimated time in seconds needed to complete a Crypto Erase sanitize operation and the associated
 *                            additional media modification in the background when the No-Deallocate After Sanitize or
 *                            the No-Deallocate Modifies Media After Sanitize features have been requested.
 */
/**
 * bd_nvme_sanitize_log_free: (skip)
 * @log: (nullable): %BDNVMESanitizeLog to free
 *
 * Frees @log.
 */
void  bd_nvme_sanitize_log_free (BDNVMESanitizeLog *log) {
        if (log == NULL)
        return;

    g_free (log);
}

/**
 * bd_nvme_sanitize_log_copy: (skip)
 * @log: (nullable): %BDNVMESanitizeLog to copy
 *
 * Creates a new copy of @log.
 */
BDNVMESanitizeLog * bd_nvme_sanitize_log_copy (BDNVMESanitizeLog *log) {
        BDNVMESanitizeLog *new_log;

    if (log == NULL)
        return NULL;

    new_log = g_new0 (BDNVMESanitizeLog, 1);
    memcpy (new_log, log, sizeof (BDNVMESanitizeLog));

    return new_log;
}

GType  bd_nvme_sanitize_log_get_type () {
        static GType type = 0;

    if (G_UNLIKELY (type == 0)) {
        type = g_boxed_type_register_static ("BDNVMESanitizeLog",
                                             (GBoxedCopyFunc) bd_nvme_sanitize_log_copy,
                                             (GBoxedFreeFunc) bd_nvme_sanitize_log_free);
    }
    return type;
}

static gboolean  bd_nvme_is_tech_avail_stub (BDNVMETech tech G_GNUC_UNUSED, G_GNUC_UNUSED guint64 mode G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_is_tech_avail' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_is_tech_avail' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_is_tech_avail) (BDNVMETech tech, G_GNUC_UNUSED guint64 mode, GError **error) = bd_nvme_is_tech_avail_stub;

/**
 * bd_nvme_is_tech_avail:
 * @tech: the queried tech
 * @mode: a bit mask of queried modes of operation (#BDNVMETechMode) for @tech
 * @error: (out) (nullable): place to store error (details about why the @tech-@mode combination is not available)
 *
 * Returns: whether the @tech-@mode combination is available -- supported by the
 *          plugin implementation and having all the runtime dependencies available
 */
gboolean  bd_nvme_is_tech_avail (BDNVMETech tech, G_GNUC_UNUSED guint64 mode, GError **error) {
    return _bd_nvme_is_tech_avail (tech, mode, error);
}


static const gchar * bd_nvme_self_test_result_to_string_stub (BDNVMESelfTestResult result G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_self_test_result_to_string' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_self_test_result_to_string' called, but not implemented!");
    return NULL;
}

static const gchar * (*_bd_nvme_self_test_result_to_string) (BDNVMESelfTestResult result, GError **error) = bd_nvme_self_test_result_to_string_stub;

/**
 * bd_nvme_self_test_result_to_string:
 * @result: A %BDNVMESelfTestResult.
 * @error: (out) (optional): place to store error (if any)
 *
 * Returns: (transfer none): A string representation of @result for use as an identifier string
 *                           or %NULL when the code is unknown.
 */
const gchar * bd_nvme_self_test_result_to_string (BDNVMESelfTestResult result, GError **error) {
    return _bd_nvme_self_test_result_to_string (result, error);
}


static BDNVMEControllerInfo * bd_nvme_get_controller_info_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_controller_info' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_controller_info' called, but not implemented!");
    return NULL;
}

static BDNVMEControllerInfo * (*_bd_nvme_get_controller_info) (const gchar *device, GError **error) = bd_nvme_get_controller_info_stub;

/**
 * bd_nvme_get_controller_info:
 * @device: a NVMe controller device (e.g. `/dev/nvme0`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves information about the NVMe controller (the Identify Controller command)
 * as specified by the @device block device path.
 *
 * Returns: (transfer full): information about given controller or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMEControllerInfo * bd_nvme_get_controller_info (const gchar *device, GError **error) {
    return _bd_nvme_get_controller_info (device, error);
}


static BDNVMENamespaceInfo * bd_nvme_get_namespace_info_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_namespace_info' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_namespace_info' called, but not implemented!");
    return NULL;
}

static BDNVMENamespaceInfo * (*_bd_nvme_get_namespace_info) (const gchar *device, GError **error) = bd_nvme_get_namespace_info_stub;

/**
 * bd_nvme_get_namespace_info:
 * @device: a NVMe namespace device (e.g. `/dev/nvme0n1`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves information about the NVMe namespace (the Identify Namespace command)
 * as specified by the @device block device path.
 *
 * Returns: (transfer full): information about given namespace or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMENamespaceInfo * bd_nvme_get_namespace_info (const gchar *device, GError **error) {
    return _bd_nvme_get_namespace_info (device, error);
}


static BDNVMESmartLog * bd_nvme_get_smart_log_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_smart_log' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_smart_log' called, but not implemented!");
    return NULL;
}

static BDNVMESmartLog * (*_bd_nvme_get_smart_log) (const gchar *device, GError **error) = bd_nvme_get_smart_log_stub;

/**
 * bd_nvme_get_smart_log:
 * @device: a NVMe controller device (e.g. `/dev/nvme0`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves drive SMART and general health information (Log Identifier `02h`).
 * The information provided is over the life of the controller and is retained across power cycles.
 *
 * Returns: (transfer full): health log data or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMESmartLog * bd_nvme_get_smart_log (const gchar *device, GError **error) {
    return _bd_nvme_get_smart_log (device, error);
}


static BDNVMEErrorLogEntry ** bd_nvme_get_error_log_entries_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_error_log_entries' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_error_log_entries' called, but not implemented!");
    return NULL;
}

static BDNVMEErrorLogEntry ** (*_bd_nvme_get_error_log_entries) (const gchar *device, GError **error) = bd_nvme_get_error_log_entries_stub;

/**
 * bd_nvme_get_error_log_entries:
 * @device: a NVMe controller device (e.g. `/dev/nvme0`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves Error Information Log (Log Identifier `01h`) entries, used to describe
 * extended error information for a command that completed with error or to report
 * an error that is not specific to a particular command. This log is global to the
 * controller. The ordering of the entries is based on the time when the error
 * occurred, with the most recent error being returned as the first log entry.
 * As the number of entries is typically limited by the drive implementation, only
 * most recent entries are provided.
 *
 * Returns: (transfer full) (array zero-terminated=1): null-terminated list
 *          of error entries or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMEErrorLogEntry ** bd_nvme_get_error_log_entries (const gchar *device, GError **error) {
    return _bd_nvme_get_error_log_entries (device, error);
}


static BDNVMESelfTestLog * bd_nvme_get_self_test_log_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_self_test_log' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_self_test_log' called, but not implemented!");
    return NULL;
}

static BDNVMESelfTestLog * (*_bd_nvme_get_self_test_log) (const gchar *device, GError **error) = bd_nvme_get_self_test_log_stub;

/**
 * bd_nvme_get_self_test_log:
 * @device: a NVMe controller device (e.g. `/dev/nvme0`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves drive self-test log (Log Identifier `06h`). Provides the status of a self-test operation
 * in progress and the percentage complete of that operation, along with the results of the last
 * 20 device self-test operations.
 *
 * Returns: (transfer full): self-test log data or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMESelfTestLog * bd_nvme_get_self_test_log (const gchar *device, GError **error) {
    return _bd_nvme_get_self_test_log (device, error);
}


static BDNVMESanitizeLog * bd_nvme_get_sanitize_log_stub (const gchar *device G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_sanitize_log' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_sanitize_log' called, but not implemented!");
    return NULL;
}

static BDNVMESanitizeLog * (*_bd_nvme_get_sanitize_log) (const gchar *device, GError **error) = bd_nvme_get_sanitize_log_stub;

/**
 * bd_nvme_get_sanitize_log:
 * @device: a NVMe controller device (e.g. `/dev/nvme0`)
 * @error: (out) (nullable): place to store error (if any)
 *
 * Retrieves the drive sanitize status log (Log Identifier `81h`) that includes information
 * about the most recent sanitize operation and the sanitize operation time estimates.
 *
 * As advised in the NVMe specification whitepaper the host should limit polling
 * to retrieve progress of a running sanitize operations (e.g. to at most once every
 * several minutes) to avoid interfering with the progress of the sanitize operation itself.
 *
 * Returns: (transfer full): sanitize log data or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_INFO
 */
BDNVMESanitizeLog * bd_nvme_get_sanitize_log (const gchar *device, GError **error) {
    return _bd_nvme_get_sanitize_log (device, error);
}


static gboolean  bd_nvme_device_self_test_stub (const gchar *device G_GNUC_UNUSED, BDNVMESelfTestAction action G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_device_self_test' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_device_self_test' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_device_self_test) (const gchar *device, BDNVMESelfTestAction action, GError **error) = bd_nvme_device_self_test_stub;

/**
 * bd_nvme_device_self_test:
 * @device: a NVMe controller or namespace device (e.g. `/dev/nvme0`)
 * @action: self-test action to take.
 * @error: (out) (nullable): place to store error (if any)
 *
 * Initiates or aborts the Device Self-test operation on the controller or a namespace,
 * distinguished by the @device path specified. In case a controller device
 * is specified then the self-test operation would include all active namespaces.
 *
 * To abort a running operation, pass #BD_NVME_SELF_TEST_ACTION_ABORT as @action.
 * To retrieve progress of a current running operation, check the self-test log using
 * bd_nvme_get_self_test_log().
 *
 * Returns: %TRUE if the device self-test command was issued successfully,
 *          %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_MANAGE
 */
gboolean  bd_nvme_device_self_test (const gchar *device, BDNVMESelfTestAction action, GError **error) {
    return _bd_nvme_device_self_test (device, action, error);
}


static gboolean  bd_nvme_format_stub (const gchar *device G_GNUC_UNUSED, guint16 lba_data_size G_GNUC_UNUSED, guint16 metadata_size G_GNUC_UNUSED, BDNVMEFormatSecureErase secure_erase G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_format' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_format' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_format) (const gchar *device, guint16 lba_data_size, guint16 metadata_size, BDNVMEFormatSecureErase secure_erase, GError **error) = bd_nvme_format_stub;

/**
 * bd_nvme_format:
 * @device: NVMe namespace or controller device to format (e.g. `/dev/nvme0n1`)
 * @lba_data_size: desired LBA data size (i.e. a sector size) in bytes or `0` to keep current. See #BDNVMELBAFormat and bd_nvme_get_namespace_info().
 * @metadata_size: desired metadata size in bytes or `0` for default. See #BDNVMELBAFormat and bd_nvme_get_namespace_info().
 * @secure_erase: optional secure erase action to take.
 * @error: (out) (nullable): place to store error (if any)
 *
 * Performs low level format of the NVM media, destroying all data and metadata for either
 * a specific namespace or all attached namespaces to the controller. Use this command
 * to change LBA sector size. Optional secure erase method can be specified as well.
 *
 * Supported LBA data sizes for a given namespace can be listed using the bd_nvme_get_namespace_info()
 * call. In case of a special value `0` the current LBA format for a given namespace will be
 * retained. When called on a controller device the first namespace is used as a reference.
 *
 * Note that the NVMe controller may define a Format NVM attribute indicating that the format
 * operation would apply to all namespaces and a format (excluding secure erase) of any
 * namespace results in a format of all namespaces in the NVM subsystem. In such case and
 * when @device is a namespace block device the #BD_NVME_ERROR_WOULD_FORMAT_ALL_NS error
 * is returned to prevent further damage. This is then supposed to be handled by the caller
 * and bd_nvme_format() is supposed to be called on a controller device instead.
 *
 * This call blocks until the format operation has finished. To retrieve progress
 * of a current running operation, check the namespace info using bd_nvme_get_namespace_info().
 *
 * Returns: %TRUE if the format command finished successfully, %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_MANAGE
 */
gboolean  bd_nvme_format (const gchar *device, guint16 lba_data_size, guint16 metadata_size, BDNVMEFormatSecureErase secure_erase, GError **error) {
    return _bd_nvme_format (device, lba_data_size, metadata_size, secure_erase, error);
}


static gboolean  bd_nvme_sanitize_stub (const gchar *device G_GNUC_UNUSED, BDNVMESanitizeAction action G_GNUC_UNUSED, gboolean no_dealloc G_GNUC_UNUSED, gint overwrite_pass_count G_GNUC_UNUSED, guint32 overwrite_pattern G_GNUC_UNUSED, gboolean overwrite_invert_pattern G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_sanitize' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_sanitize' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_sanitize) (const gchar *device, BDNVMESanitizeAction action, gboolean no_dealloc, gint overwrite_pass_count, guint32 overwrite_pattern, gboolean overwrite_invert_pattern, GError **error) = bd_nvme_sanitize_stub;

/**
 * bd_nvme_sanitize:
 * @device: NVMe namespace or controller device to format (e.g. `/dev/nvme0n1`)
 * @action: the sanitize action to perform.
 * @no_dealloc: instruct the controller to not deallocate the affected media area.
 * @overwrite_pass_count: number of overwrite passes [1-15] or 0 for the default (16 passes).
 * @overwrite_pattern: a 32-bit pattern used for the Overwrite sanitize operation.
 * @overwrite_invert_pattern: invert the overwrite pattern between passes.
 * @error: (out) (nullable): place to store error (if any)
 *
 * Starts a sanitize operation or recovers from a previously failed sanitize operation.
 * By definition, a sanitize operation alters all user data in the NVM subsystem such
 * that recovery of any previous user data from any cache, the non-volatile media,
 * or any Controller Memory Buffer is not possible. The scope of a sanitize operation
 * is all locations in the NVM subsystem that are able to contain user data, including
 * caches, Persistent Memory Regions, and unallocated or deallocated areas of the media.
 *
 * Once started, a sanitize operation is not able to be aborted and continues after
 * a Controller Level Reset including across power cycles. Once the sanitize operation
 * has run the media affected may not be immediately ready for use unless additional
 * media modification mechanism is run. This is often vendor specific and also depends
 * on the sanitize method (@action) used. Callers to this sanitize operation should
 * set @no_dealloc to %TRUE for the added convenience.
 *
 * The controller also ignores Critical Warning(s) in the SMART / Health Information
 * log page (e.g., read only mode) and attempts to complete the sanitize operation requested.
 *
 * This call returns immediately and the actual sanitize operation is performed
 * in the background. Use bd_nvme_get_sanitize_log() to retrieve status and progress
 * of a running sanitize operation. In case a sanitize operation fails the controller
 * may restrict its operation until a subsequent sanitize operation is started
 * (i.e. retried) or an #BD_NVME_SANITIZE_ACTION_EXIT_FAILURE action is used
 * to acknowledge the failure explicitly.
 *
 * The @overwrite_pass_count, @overwrite_pattern and @overwrite_invert_pattern
 * arguments are only valid when @action is #BD_NVME_SANITIZE_ACTION_OVERWRITE.
 *
 * The sanitize operation is set to run under the Allow Unrestricted Sanitize Exit
 * mode.
 *
 * Returns: %TRUE if the format command finished successfully, %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_NVME-%BD_NVME_TECH_MODE_MANAGE
 */
gboolean  bd_nvme_sanitize (const gchar *device, BDNVMESanitizeAction action, gboolean no_dealloc, gint overwrite_pass_count, guint32 overwrite_pattern, gboolean overwrite_invert_pattern, GError **error) {
    return _bd_nvme_sanitize (device, action, no_dealloc, overwrite_pass_count, overwrite_pattern, overwrite_invert_pattern, error);
}


static gchar * bd_nvme_get_host_nqn_stub (GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_host_nqn' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_host_nqn' called, but not implemented!");
    return NULL;
}

static gchar * (*_bd_nvme_get_host_nqn) (GError **error) = bd_nvme_get_host_nqn_stub;

/**
 * bd_nvme_get_host_nqn:
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Reads the Host NQN (NVM Qualified Name) value from the global `/etc/nvme/hostnqn`
 * file. An empty string is an indication that no Host NQN has been set.
 *
 * Returns: (transfer full): the Host NQN string or an empty string if none set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gchar * bd_nvme_get_host_nqn (GError **error) {
    return _bd_nvme_get_host_nqn (error);
}


static gchar * bd_nvme_generate_host_nqn_stub (GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_generate_host_nqn' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_generate_host_nqn' called, but not implemented!");
    return NULL;
}

static gchar * (*_bd_nvme_generate_host_nqn) (GError **error) = bd_nvme_generate_host_nqn_stub;

/**
 * bd_nvme_generate_host_nqn:
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Compute new Host NQN (NVM Qualified Name) value for the current system. This
 * takes in account various system identifiers (DMI, device tree) with the goal
 * of a stable unique identifier whenever feasible.
 *
 * Returns: (transfer full): the Host NQN string or %NULL with @error set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gchar * bd_nvme_generate_host_nqn (GError **error) {
    return _bd_nvme_generate_host_nqn (error);
}


static gchar * bd_nvme_get_host_id_stub (GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_get_host_id' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_get_host_id' called, but not implemented!");
    return NULL;
}

static gchar * (*_bd_nvme_get_host_id) (GError **error) = bd_nvme_get_host_id_stub;

/**
 * bd_nvme_get_host_id:
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Reads the Host ID value from the global `/etc/nvme/hostid` file. An empty
 * string is an indication that no Host ID has been set.
 *
 * Returns: (transfer full): the Host ID string or an empty string if none set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gchar * bd_nvme_get_host_id (GError **error) {
    return _bd_nvme_get_host_id (error);
}


static gboolean  bd_nvme_set_host_nqn_stub (const gchar *host_nqn G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_set_host_nqn' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_set_host_nqn' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_set_host_nqn) (const gchar *host_nqn, GError **error) = bd_nvme_set_host_nqn_stub;

/**
 * bd_nvme_set_host_nqn:
 * @host_nqn: The Host NVM Qualified Name.
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Writes the Host NQN (NVM Qualified Name) value to the system `/etc/nvme/hostnqn` file.
 * No validation of the string is performed.
 *
 * Returns: %TRUE if the value was set successfully or %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gboolean  bd_nvme_set_host_nqn (const gchar *host_nqn, GError **error) {
    return _bd_nvme_set_host_nqn (host_nqn, error);
}


static gboolean  bd_nvme_set_host_id_stub (const gchar *host_id G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_set_host_id' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_set_host_id' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_set_host_id) (const gchar *host_id, GError **error) = bd_nvme_set_host_id_stub;

/**
 * bd_nvme_set_host_id:
 * @host_id: The Host ID.
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Writes the Host ID value to the system `/etc/nvme/hostid` file.
 * No validation of the string is performed.
 *
 * Returns: %TRUE if the value was set successfully or %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gboolean  bd_nvme_set_host_id (const gchar *host_id, GError **error) {
    return _bd_nvme_set_host_id (host_id, error);
}


static gboolean  bd_nvme_connect_stub (const gchar *subsysnqn G_GNUC_UNUSED, const gchar *transport G_GNUC_UNUSED, const gchar *transport_addr G_GNUC_UNUSED, const gchar *transport_svcid G_GNUC_UNUSED, const gchar *host_traddr G_GNUC_UNUSED, const gchar *host_iface G_GNUC_UNUSED, const gchar *host_nqn G_GNUC_UNUSED, const gchar *host_id G_GNUC_UNUSED, const BDExtraArg **extra G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_connect' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_connect' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_connect) (const gchar *subsysnqn, const gchar *transport, const gchar *transport_addr, const gchar *transport_svcid, const gchar *host_traddr, const gchar *host_iface, const gchar *host_nqn, const gchar *host_id, const BDExtraArg **extra, GError **error) = bd_nvme_connect_stub;

/**
 * bd_nvme_connect:
 * @subsysnqn: The name for the NVMe subsystem to connect to.
 * @transport: The network fabric used for a NVMe-over-Fabrics network.
 * @transport_addr: (nullable): The network address of the Controller. For transports using IP addressing (e.g. `rdma`) this should be an IP-based address.
 * @transport_svcid: (nullable): The transport service id.  For transports using IP addressing (e.g. `rdma`) this field is the port number. By default, the IP port number for the `RDMA` transport is `4420`.
 * @host_traddr: (nullable): The network address used on the host to connect to the Controller. For TCP, this sets the source address on the socket.
 * @host_iface: (nullable): The network interface used on the host to connect to the Controller (e.g. IP `eth1`, `enp2s0`). This forces the connection to be made on a specific interface instead of letting the system decide.
 * @host_nqn: (nullable): Overrides the default Host NQN that identifies the NVMe Host. If this option is %NULL, the default is read from `/etc/nvme/hostnqn` first.
 *                        If that does not exist, the autogenerated NQN value from the NVMe Host kernel module is used next. The Host NQN uniquely identifies the NVMe Host.
 * @host_id: (nullable): User-defined host UUID or %NULL to use default (as defined in `/etc/nvme/hostid`)
 * @extra: (nullable) (array zero-terminated=1): Additional arguments.
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Creates a transport connection to a remote system (specified by @transport_addr and @transport_svcid)
 * and creates a NVMe over Fabrics controller for the NVMe subsystem specified by the @subsysnqn option.
 *
 * Valid values for @transport include:
 * - `"rdma"`: An rdma network (RoCE, iWARP, Infiniband, basic rdma, etc.)
 * - `"fc"`: A Fibre Channel network.
 * - `"tcp"`: A TCP/IP network.
 * - `"loop"`: A NVMe over Fabrics target on the local host.
 *
 * In addition to the primary options it's possible to supply @extra arguments:
 * - `"config"`: Use the specified JSON configuration file instead of the default file (see below) or
 *               specify `"none"` to avoid reading any configuration file.
 * - `"dhchap_key"`: NVMe In-band authentication secret in ASCII format as described
 *                      in the NVMe 2.0 specification. When not specified, the secret is by default read
 *                      from `/etc/nvme/hostkey`. In case that file does not exist no in-band authentication
 *                      is attempted.
 * - `"dhchap_ctrl_key"`: NVMe In-band authentication controller secret for bi-directional authentication.
 *                        When not specified, no bi-directional authentication is attempted.
 * - `"nr_io_queues"`: The number of I/O queues.
 * - `"nr_write_queues"`: Number of additional queues that will be used for write I/O.
 * - `"nr_poll_queues"`: Number of additional queues that will be used for polling latency sensitive I/O.
 * - `"queue_size"`: Number of elements in the I/O queues.
 * - `"keep_alive_tmo"`: The keep alive timeout (in seconds).
 * - `"reconnect_delay"`: The delay (in seconds) before reconnect is attempted after a connect loss.
 * - `"ctrl_loss_tmo"`: The controller loss timeout period (in seconds). A special value of `-1` will cause reconnecting forever.
 * - `"fast_io_fail_tmo"`: Fast I/O Fail timeout (in seconds).
 * - `"tos"`: Type of service.
 * - `"duplicate_connect"`: Allow duplicated connections between same transport host and subsystem port. Boolean value.
 * - `"disable_sqflow"`: Disables SQ flow control to omit head doorbell update for submission queues when sending nvme completions. Boolean value.
 * - `"hdr_digest"`: Generates/verifies header digest (TCP). Boolean value.
 * - `"data_digest"`: Generates/verifies data digest (TCP). Boolean value.
 * - `"tls"`: Enable TLS encryption (TCP). Boolean value.
 * - `"hostsymname"`: TP8010: NVMe host symbolic name.
 * - `"keyring"`: Keyring to store and lookup keys. String value.
 * - `"tls_key"`: TLS PSK for the connection. String value.
 *
 * Boolean values can be expressed by "0"/"1", "on"/"off" or "True"/"False" case-insensitive
 * strings. Failed numerical or boolean string conversions will result in the option being ignored.
 *
 * By default additional options are read from the default configuration file `/etc/nvme/config.json`.
 * This follows the default behaviour of `nvme-cli`. Use the @extra `"config"` argument
 * to either specify a different config file or disable use of it. The JSON configuration
 * file format is documented in [https://raw.githubusercontent.com/linux-nvme/libnvme/master/doc/config-schema.json](https://raw.githubusercontent.com/linux-nvme/libnvme/master/doc/config-schema.json).
 * As a rule @extra key names are kept consistent with the JSON config file schema.
 * Any @extra option generally overrides particular option specified in a configuration file.
 *
 * Returns: %TRUE if the subsystem was connected successfully, %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gboolean  bd_nvme_connect (const gchar *subsysnqn, const gchar *transport, const gchar *transport_addr, const gchar *transport_svcid, const gchar *host_traddr, const gchar *host_iface, const gchar *host_nqn, const gchar *host_id, const BDExtraArg **extra, GError **error) {
    return _bd_nvme_connect (subsysnqn, transport, transport_addr, transport_svcid, host_traddr, host_iface, host_nqn, host_id, extra, error);
}


static gboolean  bd_nvme_disconnect_stub (const gchar *subsysnqn G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_disconnect' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_disconnect' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_disconnect) (const gchar *subsysnqn, GError **error) = bd_nvme_disconnect_stub;

/**
 * bd_nvme_disconnect:
 * @subsysnqn: The name of the NVMe subsystem to disconnect.
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Disconnects and removes one or more existing NVMe over Fabrics controllers.
 * This may disconnect multiple controllers with matching @subsysnqn and %TRUE
 * is only returned when all controllers were disconnected successfully.
 *
 * Returns: %TRUE if all matching controllers were disconnected successfully, %FALSE with @error
 *          set in case of a disconnect error or when no matching controllers were found.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gboolean  bd_nvme_disconnect (const gchar *subsysnqn, GError **error) {
    return _bd_nvme_disconnect (subsysnqn, error);
}


static gboolean  bd_nvme_disconnect_by_path_stub (const gchar *path G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_disconnect_by_path' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_disconnect_by_path' called, but not implemented!");
    return FALSE;
}

static gboolean  (*_bd_nvme_disconnect_by_path) (const gchar *path, GError **error) = bd_nvme_disconnect_by_path_stub;

/**
 * bd_nvme_disconnect_by_path:
 * @path: NVMe controller device to disconnect (e.g. `/dev/nvme0`).
 * @error: (out) (nullable): Place to store error (if any).
 *
 * Disconnects and removes a NVMe over Fabrics controller represented
 * by a block device path.
 *
 * Returns: %TRUE if the controller was disconnected successfully,
 *          %FALSE otherwise with @error set.
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gboolean  bd_nvme_disconnect_by_path (const gchar *path, GError **error) {
    return _bd_nvme_disconnect_by_path (path, error);
}


static gchar ** bd_nvme_find_ctrls_for_ns_stub (const gchar *ns_sysfs_path G_GNUC_UNUSED, const gchar *subsysnqn G_GNUC_UNUSED, const gchar *host_nqn G_GNUC_UNUSED, const gchar *host_id G_GNUC_UNUSED, GError **error) {
    bd_utils_log_format (BD_UTILS_LOG_CRIT, "The function 'bd_nvme_find_ctrls_for_ns' called, but not implemented!");
    g_set_error (error, BD_INIT_ERROR, BD_INIT_ERROR_NOT_IMPLEMENTED,
                "The function 'bd_nvme_find_ctrls_for_ns' called, but not implemented!");
    return NULL;
}

static gchar ** (*_bd_nvme_find_ctrls_for_ns) (const gchar *ns_sysfs_path, const gchar *subsysnqn, const gchar *host_nqn, const gchar *host_id, GError **error) = bd_nvme_find_ctrls_for_ns_stub;

/**
 * bd_nvme_find_ctrls_for_ns:
 * @ns_sysfs_path: NVMe namespace device file.
 * @subsysnqn: (nullable): Limit matching to the specified subsystem NQN.
 * @host_nqn: (nullable): Limit matching to the specified host NQN.
 * @host_id: (nullable): Limit matching to the specified host ID.
 * @error: (out) (nullable): Place to store error (if any).
 *
 * A convenient utility function to look up all controllers associated
 *  with a NVMe subsystem the specified namespace is part of.
 *
 * Returns: (transfer full) (array zero-terminated=1): list of controller sysfs paths
 *          or %NULL in case of an error (with @error set).
 *
 * Tech category: %BD_NVME_TECH_FABRICS-%BD_NVME_TECH_MODE_INITIATOR
 */
gchar ** bd_nvme_find_ctrls_for_ns (const gchar *ns_sysfs_path, const gchar *subsysnqn, const gchar *host_nqn, const gchar *host_id, GError **error) {
    return _bd_nvme_find_ctrls_for_ns (ns_sysfs_path, subsysnqn, host_nqn, host_id, error);
}


static gpointer load_nvme_from_plugin(const gchar *so_name) {
    void *handle = NULL;
    char *error = NULL;
    gboolean (*init_fn) (void) = NULL;

    handle = dlopen(so_name, RTLD_LAZY);
    if (!handle) {
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load module nvme: %s", dlerror());
        return NULL;
    }

    dlerror();
    * (void**) (&init_fn) = dlsym(handle, "bd_nvme_init");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_DEBUG, "failed to load the init() function for nvme: %s", error);
    /* coverity[dead_error_condition] */
    if (init_fn && !init_fn()) {
        dlclose(handle);
        return NULL;
    }
    init_fn = NULL;

    dlerror();
    * (void**) (&_bd_nvme_is_tech_avail) = dlsym(handle, "bd_nvme_is_tech_avail");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_is_tech_avail: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_self_test_result_to_string) = dlsym(handle, "bd_nvme_self_test_result_to_string");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_self_test_result_to_string: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_controller_info) = dlsym(handle, "bd_nvme_get_controller_info");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_controller_info: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_namespace_info) = dlsym(handle, "bd_nvme_get_namespace_info");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_namespace_info: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_smart_log) = dlsym(handle, "bd_nvme_get_smart_log");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_smart_log: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_error_log_entries) = dlsym(handle, "bd_nvme_get_error_log_entries");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_error_log_entries: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_self_test_log) = dlsym(handle, "bd_nvme_get_self_test_log");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_self_test_log: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_sanitize_log) = dlsym(handle, "bd_nvme_get_sanitize_log");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_sanitize_log: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_device_self_test) = dlsym(handle, "bd_nvme_device_self_test");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_device_self_test: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_format) = dlsym(handle, "bd_nvme_format");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_format: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_sanitize) = dlsym(handle, "bd_nvme_sanitize");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_sanitize: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_host_nqn) = dlsym(handle, "bd_nvme_get_host_nqn");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_host_nqn: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_generate_host_nqn) = dlsym(handle, "bd_nvme_generate_host_nqn");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_generate_host_nqn: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_get_host_id) = dlsym(handle, "bd_nvme_get_host_id");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_get_host_id: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_set_host_nqn) = dlsym(handle, "bd_nvme_set_host_nqn");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_set_host_nqn: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_set_host_id) = dlsym(handle, "bd_nvme_set_host_id");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_set_host_id: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_connect) = dlsym(handle, "bd_nvme_connect");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_connect: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_disconnect) = dlsym(handle, "bd_nvme_disconnect");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_disconnect: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_disconnect_by_path) = dlsym(handle, "bd_nvme_disconnect_by_path");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_disconnect_by_path: %s", error);

    dlerror();
    * (void**) (&_bd_nvme_find_ctrls_for_ns) = dlsym(handle, "bd_nvme_find_ctrls_for_ns");
    if ((error = dlerror()) != NULL)
        bd_utils_log_format (BD_UTILS_LOG_WARNING, "failed to load bd_nvme_find_ctrls_for_ns: %s", error);

    return handle;
}

static gboolean unload_nvme (gpointer handle) {
    char *error = NULL;
    gboolean (*close_fn) (void) = NULL;

    _bd_nvme_is_tech_avail = bd_nvme_is_tech_avail_stub;
    _bd_nvme_self_test_result_to_string = bd_nvme_self_test_result_to_string_stub;
    _bd_nvme_get_controller_info = bd_nvme_get_controller_info_stub;
    _bd_nvme_get_namespace_info = bd_nvme_get_namespace_info_stub;
    _bd_nvme_get_smart_log = bd_nvme_get_smart_log_stub;
    _bd_nvme_get_error_log_entries = bd_nvme_get_error_log_entries_stub;
    _bd_nvme_get_self_test_log = bd_nvme_get_self_test_log_stub;
    _bd_nvme_get_sanitize_log = bd_nvme_get_sanitize_log_stub;
    _bd_nvme_device_self_test = bd_nvme_device_self_test_stub;
    _bd_nvme_format = bd_nvme_format_stub;
    _bd_nvme_sanitize = bd_nvme_sanitize_stub;
    _bd_nvme_get_host_nqn = bd_nvme_get_host_nqn_stub;
    _bd_nvme_generate_host_nqn = bd_nvme_generate_host_nqn_stub;
    _bd_nvme_get_host_id = bd_nvme_get_host_id_stub;
    _bd_nvme_set_host_nqn = bd_nvme_set_host_nqn_stub;
    _bd_nvme_set_host_id = bd_nvme_set_host_id_stub;
    _bd_nvme_connect = bd_nvme_connect_stub;
    _bd_nvme_disconnect = bd_nvme_disconnect_stub;
    _bd_nvme_disconnect_by_path = bd_nvme_disconnect_by_path_stub;
    _bd_nvme_find_ctrls_for_ns = bd_nvme_find_ctrls_for_ns_stub;

    dlerror();
    * (void**) (&close_fn) = dlsym(handle, "bd_nvme_close");
    if (((error = dlerror()) != NULL) || !close_fn)
        bd_utils_log_format (BD_UTILS_LOG_DEBUG, "failed to load the close_plugin() function for nvme: %s", error);
    /* coverity[dead_error_condition] */
    if (close_fn) {
        close_fn();
    }

    return dlclose(handle) == 0;
}

