/*
 * (C) Copyright 2005- ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 *
 * In applying this licence, ECMWF does not waive the privileges and immunities granted to it by
 * virtue of its status as an intergovernmental organisation nor does it submit to any jurisdiction.
 */

#include "Bit.h"

eccodes::accessor::Bit _grib_accessor_bit;
eccodes::Accessor* grib_accessor_bit = &_grib_accessor_bit;

namespace eccodes::accessor
{

void Bit::init(const long len, grib_arguments* arg)
{
    Long::init(len, arg);
    length_    = 0;
    owner_     = arg->get_name(get_enclosing_handle(), 0);
    bit_index_ = arg->get_long(get_enclosing_handle(), 1);
}

int Bit::unpack_long(long* val, size_t* len)
{
    int ret   = 0;
    long data = 0;

    if (*len < 1) {
        grib_context_log(context_, GRIB_LOG_ERROR, "Bit: unpack_long: Wrong size for %s, it contains %d values ", name_, 1);
        *len = 1;
        return GRIB_ARRAY_TOO_SMALL;
    }

    if ((ret = grib_get_long_internal(get_enclosing_handle(), owner_, &data)) != GRIB_SUCCESS) {
        *len = 0;
        return ret;
    }

    if (data & (1 << bit_index_))
        *val = 1;
    else
        *val = 0;

    *len = 1;
    return GRIB_SUCCESS;
}

int Bit::pack_long(const long* val, size_t* len)
{
    if (*len < 1) {
        grib_context_log(context_, GRIB_LOG_ERROR, "Bit: pack_long: At least one value to pack for %s", name_);
        *len = 1;
        return GRIB_ARRAY_TOO_SMALL;
    }

    grib_accessor* owner = grib_find_accessor(get_enclosing_handle(), owner_);
    if (!owner) {
        grib_context_log(context_, GRIB_LOG_ERROR, "Bit: Cannot get the owner %s for computing the bit value of %s",
                         owner_, name_);
        *len = 0;
        return GRIB_NOT_FOUND;
    }

    unsigned char* mdata = get_enclosing_handle()->buffer->data;
    mdata += owner->byte_offset();
    /* Note: In the definitions, flagbit numbers go from 7 to 0 (the bit_index), while WMO convention is from 1 to 8 */
    if (context_->debug) {
        /* Print bit positions from 1 (MSB) */
        fprintf(stderr, "ECCODES DEBUG Setting bit %d in %s to %d\n", 8 - bit_index_, owner->name_, (*val > 0));
    }
    grib_set_bit(mdata, 7 - bit_index_, *val > 0);

    *len = 1;
    return GRIB_SUCCESS;
}

}  // namespace eccodes::accessor
