/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
 */
package gov.nasa.worldwind.servers.wms.generators;

import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.servers.wms.*;
import gov.nasa.worldwind.servers.wms.formats.BufferedImageFormatter;
import gov.nasa.worldwind.servers.wms.formats.ImageFormatter;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Properties;
import javax.imageio.ImageIO;

/**
 * A MapGenerator implementation for serving high-resolution images from NASA's Blue Marble,
 * Next-Generation image series.
 * <p>
 * It is assumed that the source PNG files for this dataseries, as distributed by NASA, have
 * been converted to geotiffs, with georeferencing explicitly injected into the files. It is
 * also highly recommended, but not required, that the geotiffs have a subtiling organization.
 * All of these steps can be conveniently performed with the GDAL utility <code>gdal_translate</code>.
 * </p>
 * <p>
 * This generator also uses the GDAL utility <code>gdalwarp</code> to extract subregions from
 * the source images.
 * </p>
 * <p/>
 * <p>The NASA source files have a regular naming scheme that encodes the date and
 * georeferencing information. The files are thus otherwise named with a prefix that
 * indicates the features depicted in the image (topography, bathyrmetry, etc.). The naming
 * scheme looks like:
 * </p>
 * <pre>
 *      <i><b>name_prefix</b></i>.<i><b>yyyymm</b></i>.3x21600x21600.A1.<i><b>suffix</b></i>
 *      <i><b>name_prefix</b></i>.<i><b>yyyymm</b></i>.3x21600x21600.A2.<i><b>suffix</b></i>
 *      <i><b>name_prefix</b></i>.<i><b>yyyymm</b></i>.3x21600x21600.B1.<i><b>suffix</b></i>
 *      <i><b>name_prefix</b></i>.<i><b>yyyymm</b></i>.3x21600x21600.B2.<i><b>suffix</b></i>
 *      ...
 *      <i><b>name_prefix</b></i>.<i><b>yyyymm</b></i>.3x21600x21600.D2.<i><b>suffix</b></i>
 * </pre>
 * <p>
 * The substring "yyyymm" indicates the date of the imagery. The codes A1, A2, ..., D1, D2 reflect
 * an implicit goereferencing. The high-resolution images are distributed as a set
 * of 8 90x90 degree tiles. The "1" indicates northern hemisphere (i.e., the latitude for that
 * image ranges from 0N -- 90N), whereas "2" indicates southern hemisphere (0-90S). The letter
 * codes indicate bounds in longitude:</p>
 * <pre>
 *     A = 180W --  90W
 *     B =  90W --   0
 *     C =   0  --  90E
 *     D =  90E -- 180E
 * </pre>
 * <p>
 * There are several required properties needed in the configuration of the
 * {@link gov.nasa.worldwind.servers.wms.MapSource} element that reflect the italicized
 * substrings in the above naming scheme, for the actual filenames of the geotiffs to be served:
 * </p>
 * <pre>
 *   &lt;!-- name_prefix --&gt;
 *   &lt;property name="BlueMarble500M.namingscheme.prefix" value="..." /&gt;
 * <p/>
 *   &lt;!-- suffix --&gt;
 *   &lt;property name="BlueMarble500M.namingscheme.suffix" value="..." /&gt;
 * <p/>
 *   &lt;!-- yyyymm --&gt;
 *   &lt;property name="BlueMarble500M.defaultTime" value="..." /&gt;
 * </pre>
 *
 * @author brownrigg
 * @version $Id$
 */

public class BlueMarbleNG500MGenerator implements MapGenerator {

    public BlueMarbleNG500MGenerator() {
    }

    public ServiceInstance getServiceInstance() {
        return new BMNGServiceInstance();
    }

    public boolean initialize(MapSource mapSource) throws IOException, WMSServiceException {
        boolean success = true;  // Assume the best until proven otherwise...
        try {
            this.mapSource = mapSource;
            this.rootDir = mapSource.getRootDir();

            this.gdalPath = WMSServer.getConfiguration().getGDALPath();
            if (this.gdalPath == null)
                this.gdalPath = "";
            else if (!this.gdalPath.endsWith(File.separator))
                this.gdalPath += File.separator;   // make this well-formed for convenience later...

            // Extract expected properties that should have been set in our MapSource
            // configuration...
            Properties myProps = mapSource.getProperties();
            if (myProps == null)
                throw new IllegalArgumentException("Missing properties in configuration for MapSource: "
                        + mapSource.getServiceClass().getName());

            this.defaultMonth = myProps.getProperty(DEFAULT_DATASET);
            this.namePrefix = myProps.getProperty(FILE_PREFIX);
            this.nameSuffix = myProps.getProperty(FILE_SUFFIX);

            if (this.defaultMonth == null || this.namePrefix == null || this.nameSuffix == null) {
                StringBuilder errMsg = new StringBuilder();
                errMsg.append("invalid properties file\n");
                errMsg.append(DEFAULT_DATASET).append(" = ").append(this.defaultMonth).append("\n");
                errMsg.append(FILE_PREFIX).append(" = ").append(this.namePrefix).append("\n");
                errMsg.append(FILE_SUFFIX).append(" = ").append(this.nameSuffix).append("\n");
                throw new IllegalArgumentException(errMsg.toString());
            }

        } catch (Exception ex) {
            SysLog.inst().error("BlueMarbleNG500MGenerator initialization failed: " + ex.getMessage());
            success = false;
        }

        return success;
    }

    public Sector getBBox() {
        return new Sector(
                Angle.fromDegreesLatitude(-90.),
                Angle.fromDegreesLatitude(90.),
                Angle.fromDegreesLongitude(-180.),
                Angle.fromDegreesLongitude(180.));

    }

    public String[] getCRS() {
        return new String[]{crsStr};
    }


    private class BMNGServiceInstance implements ServiceInstance {
        public ImageFormatter serviceRequest(WMSGetMapRequest req) throws IOException, WMSServiceException {

            // the image to be created...
            BufferedImage reqImage = new BufferedImage(req.getWidth(), req.getHeight(), BufferedImage.TYPE_4BYTE_ABGR);
            Graphics2D g2d = (Graphics2D) reqImage.getGraphics();

            // Figure out what parts of the BMNG grid the request overlaps...
            Sector reqSector = Sector.fromDegrees(req.getBBoxYMin(), req.getBBoxYMax(), req.getBBoxXMin(),
                    req.getBBoxXMax());
            int[] lonIndices = pigeonHoleLon(req.getBBoxXMin(), req.getBBoxXMax());
            int[] latIndices = pigeonHoleLat(req.getBBoxYMin(), req.getBBoxYMax());

            // Extract source images from overlapped tiles...
            for (int iLonCell = lonIndices[0]; iLonCell <= lonIndices[1]; iLonCell++) {
                for (int iLatCell = latIndices[0]; iLatCell <= latIndices[1]; iLatCell++) {
                    // compute request overlap with the BMNG tile...
                    double minLon = (req.getBBoxXMin() <= lonBounds[iLonCell]) ? lonBounds[iLonCell] : req.getBBoxXMin();
                    double maxLon = (req.getBBoxXMax() >= lonBounds[iLonCell + 1]) ? lonBounds[iLonCell + 1] : req.getBBoxXMax();
                    double minLat = (req.getBBoxYMin() <= latBounds[iLatCell]) ? latBounds[iLatCell] : req.getBBoxYMin();
                    double maxLat = (req.getBBoxYMax() >= latBounds[iLatCell + 1]) ? latBounds[iLatCell + 1] : req.getBBoxYMax();
                    Sector tileSector = Sector.fromDegrees(minLat, maxLat, minLon, maxLon);
                    Sector overlap = reqSector.intersection(tileSector);
                    if (overlap == null) {
                        continue;
                    }

                    // compute name of BMNG tile...
                    StringBuilder source = new StringBuilder(rootDir);
                    source.append(File.separator).append(namePrefix).append(".").
                            append(defaultMonth).append(".").
                            append(BMNG_NAME_CONSTANT).append(".").
                            append(lonCodes[iLonCell]).append(latCodes[iLatCell]).append(".").
                            append(nameSuffix);

                    // footprint of this tile in the destination image...
                    int dx1 = (int) ((overlap.getMinLongitude().degrees - reqSector.getMinLongitude().degrees) * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                    int dx2 = (int) ((overlap.getMaxLongitude().degrees - reqSector.getMinLongitude().degrees) * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                    int dy1 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMaxLatitude().degrees) * reqImage.getHeight() / reqSector.getDeltaLatDegrees());
                    int dy2 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMinLatitude().degrees) * reqImage.getHeight() / reqSector.getDeltaLatDegrees());

                    // hand off to GDAL to do the work...
                    BufferedImage sourceImage = getImageFromSource(source.toString(), overlap, (dx2 - dx1), (dy2 - dy1));
                    if (sourceImage == null) continue;

                    g2d.drawImage(sourceImage, dx1, dy1, dx2, dy2, 0, 0, sourceImage.getWidth(),
                            sourceImage.getHeight(), null);
                }
            }

            return new BufferedImageFormatter(reqImage);
        }

        // Not implemented at present.
        public java.util.List<File> serviceRequest(WMSGetImageryListRequest req) throws IOException, WMSServiceException {
            return null;
        }

        public void freeResources() {
        }

        //
        // Attempts to return the specified image as a BufferedImage. Returns null on failure.
        //
        private BufferedImage getImageFromSource(String tilename, Sector extent, int xres, int yres)
                throws WMSServiceException
        {
            BufferedImage sourceImage = null;
            File tmpFile = new TempFile(getTempFilename());
            try {
                StringBuilder cmd = new StringBuilder();
                cmd.append(gdalPath);

                if (extent.getDeltaLon().getDegrees() < FILTER_THRESHOLD ||
                        extent.getDeltaLat().getDegrees() < FILTER_THRESHOLD) {
                    // We use gdalwarp at these larger-scale requests, as it has better filtering behaviors...
                    cmd.append("gdalwarp").
                            append(" -of Gtiff").append(" -te ").
                            append(extent.getMinLongitude()).append(" ").
                            append(extent.getMinLatitude()).append(" ").
                            append(extent.getMaxLongitude()).append(" ").
                            append(extent.getMaxLatitude()).append(" ");
                    cmd.append(" -ts ").
                            append(xres).append(" ").
                            append(yres).append(" ");
                    cmd.append(" -rc ");  // cubic filtering
                    cmd.append(new File(tilename).getAbsolutePath()).append(" ").
                            append(tmpFile.getAbsolutePath());
                } else {
                    // We use gdal_translate at smaller scale requests, as it can make use of embedded
                    // overview images if present...
                    cmd.append("gdal_translate -not_strict").
                            append(" -of png").append(" -projwin ").
                            append(extent.getMinLongitude()).append(" ").
                            append(extent.getMaxLatitude()).append(" ").
                            append(extent.getMaxLongitude()).append(" ").
                            append(extent.getMinLatitude()).append(" ");
                    cmd.append(" -outsize ").
                            append(xres).append(" ").
                            append(yres).append(" ");
                    cmd.append(new File(tilename).getAbsolutePath()).append(" ").
                            append(tmpFile.getAbsolutePath());
                }

                SysLog.inst().info("Thread " + threadId + ", exec'ing: " + cmd.toString());

                // Call upon a GDAL utility to make our map...
                Process proc = null;
                try {
                    proc = Runtime.getRuntime().exec(cmd.toString());
                    proc.waitFor();
                    if (proc.exitValue() != 0) {
                        // fetch enough of the stdout/stderr to give a hint...
                        byte[] buff = new byte[1024];
                        int len = proc.getInputStream().read(buff);
                        if (len <= 0)   // try stderr?
                            len = proc.getErrorStream().read(buff);
                        throw new WMSServiceException("Thread " + threadId + ", failed to execute GDAL: " +
                                ((len > 0) ? new String(buff, 0, len) : "**unknown**"));
                    }

                    if (null != tmpFile)
                    {
                        if( tmpFile.exists() )
                            sourceImage = ImageIO.read(tmpFile);
                        else
                            SysLog.inst().info("tmpFile " + tmpFile.getAbsoluteFile() + " does not exist" );
                    }
                    else
                        SysLog.inst().info("tmpFile is null. why?? ");

                } catch (InterruptedException ex) {
                    throw new WMSServiceException(ex.toString());
                } finally {
                    try {
                        // Note that we actively try to clean up here, but have a passive mechanism also in the
                        // TempFile finalizer. This is all an effort to keep these tempfiles from lingering around
                        // and remaining if the server is stopped.
                        if (null != tmpFile) {
                            tmpFile.delete();
                        }
                    } catch (Exception ex) {/* best effort */}
                    try {
                        if (null != proc) {
                            proc.getInputStream().close();
                            proc.getErrorStream().close();
                            proc.getOutputStream().close();
                        }
                    } catch (Exception ex) {/* best effort */}
                }

            } catch (Exception ex) {
                throw new WMSServiceException(ex.toString());
            }

            return sourceImage;
        }

        private String getTempFilename() {
            StringBuilder tmp = new StringBuilder();
            tmp.append(WMSServer.getConfiguration().getWorkDirectory()).
                    append(File.separator).append("WMStmp").
                    append(Integer.toString((int) (Math.random() * 100000.)));
            return tmp.toString();
        }

        // Pigeon-holing note:  Requests that have one of the bbox min/max's fall on the BMNG 2x4 grid boundaries
        // are problemmatic, as which cell does that bbox edge lie in (?) The rule used here is that the request min's'
        // are assigned to the highest-number cell they can fit it, whereas the max's are placed in the lowest
        // numbered cell.

        private int[] pigeonHoleLon(double min, double max) {
            min = (min < -180.) ? -180. : min;
            max = (max > 180.) ? 180. : max;
            int[] ret = new int[2];
            ret[0] = 0;
            ret[1] = lonBounds.length - 2;

            for (int i = lonBounds.length - 1; i > 0; i--) {
                if (min >= lonBounds[i - 1] && min <= lonBounds[i]) {
                    ret[0] = i - 1;
                    break;
                }
            }

            for (int i = 1; i < lonBounds.length; i++) {
                if (max >= lonBounds[i - 1] && max <= lonBounds[i]) {
                    ret[1] = i - 1;
                    break;
                }
            }

            return ret;
        }

        private int[] pigeonHoleLat(double min, double max) {
            min = (min < -90.) ? -90. : min;
            max = (max > 90.) ? 90. : max;
            int[] ret = new int[2];
            ret[0] = 0;
            ret[1] = latBounds.length - 2;

            for (int i = latBounds.length - 1; i > 0; i--) {
                if (min >= latBounds[i - 1] && min <= latBounds[i]) {
                    ret[0] = i - 1;
                    break;
                }
            }

            for (int i = 1; i < latBounds.length; i++) {
                if (max >= latBounds[i - 1] && max <= latBounds[i]) {
                    ret[1] = i - 1;
                    break;
                }
            }

            return ret;
        }

       private long threadId;  // Used as part of logging...
    }


    private MapSource mapSource = null;
    private String rootDir = null;
    private String defaultMonth = null;
    private String nameSuffix = null;
    private String namePrefix = null;
    private String gdalPath = null;

    private static final double[] lonBounds = {-180., -90., 0., 90., 180.};
    private static final double[] latBounds = {-90., 0., 90.};
    private static final String[] lonCodes = {"A", "B", "C", "D"};
    private static final String[] latCodes = {"2", "1"};
    private static final String crsStr = "EPSG:4326";
    private static final String DEFAULT_DATASET = "BlueMarble500M.defaultTime";
    private static final String FILE_PREFIX = "BlueMarble500M.namingscheme.prefix";
    private static final String FILE_SUFFIX = "BlueMarble500M.namingscheme.suffix";
    private static final String BMNG_NAME_CONSTANT = "3x21600x21600";
    private static final double FILTER_THRESHOLD = 5.0;

}
