"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _configSchema = require("@osd/config-schema");

var _common = require("../../common");

var _createReport = require("./lib/createReport");

var _helpers = require("./utils/helpers");

var _constants = require("./utils/constants");

var _backendToUi = require("./utils/converters/backendToUi");

var _metricHelper = require("./utils/metricHelper");

var _validationHelper = require("../../server/utils/validationHelper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function _default(router, config) {
  const protocol = config.get('osd_server', 'protocol');
  const hostname = config.get('osd_server', 'hostname');
  const port = config.get('osd_server', 'port');
  const basePath = config.osdConfig.get('server', 'basePath'); // generate report (with provided metadata)

  router.post({
    path: `${_common.API_PREFIX}/generateReport`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.maybe(_configSchema.schema.string()),
        dateFormat: _configSchema.schema.maybe(_configSchema.schema.string()),
        csvSeparator: _configSchema.schema.maybe(_configSchema.schema.string()),
        allowLeadingWildcards: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;
    let report = request.body; // input validation

    try {
      report.report_definition.report_params.core_params.origin = `${protocol}://${hostname}:${port}${basePath}`;
      report = await (0, _validationHelper.validateReport)(context.core.opensearch.legacy.client, report, basePath);
    } catch (error) {
      logger.error(`Failed input validation for create report ${error}`);
      (0, _metricHelper.addToMetric)('report', 'create', 'user_error');
      return response.badRequest({
        body: error
      });
    }

    try {
      const reportData = await (0, _createReport.createReport)(request, context, report, config); // if not deliver to user himself , no need to send actual file data to client

      const delivery = report.report_definition.delivery;
      (0, _metricHelper.addToMetric)('report', 'create', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      // TODO: better error handling for delivery and stages in generating report, pass logger to deeper level
      logger.error(`Failed to generate report: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // generate report from report id

  router.get({
    path: `${_common.API_PREFIX}/generateReport/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string(),
        allowLeadingWildcards: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'download', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;

    try {
      const savedReportId = request.params.reportId; // @ts-ignore

      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request); // get report

      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: savedReportId
      }); // convert report to use UI model

      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath); // generate report

      const reportData = await (0, _createReport.createReport)(request, context, report, config, savedReportId);
      (0, _metricHelper.addToMetric)('report', 'download', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report by id: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'download', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // create report from existing report definition

  router.post({
    path: `${_common.API_PREFIX}/generateReport/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string(),
        allowLeadingWildcards: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;
    const reportDefinitionId = request.params.reportDefinitionId;
    let report;

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request); // call OpenSearch API to create report from definition

      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.createReportFromDefinition', {
        reportDefinitionId: reportDefinitionId,
        body: {
          reportDefinitionId: reportDefinitionId
        }
      });
      const reportId = opensearchResp.reportInstance.id; // convert report to use UI model

      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath); // generate report

      const reportData = await (0, _createReport.createReport)(request, context, report, config, reportId);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report from reportDefinition id ${reportDefinitionId} : ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get all reports details

  router.get({
    path: `${_common.API_PREFIX}/reports`,
    validate: {
      query: _configSchema.schema.object({
        fromIndex: _configSchema.schema.maybe(_configSchema.schema.number()),
        maxItems: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'list', 'count');
    const {
      fromIndex,
      maxItems
    } = request.query;

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReports', {
        fromIndex: fromIndex,
        maxItems: maxItems || _constants.DEFAULT_MAX_SIZE
      });
      const reportsList = (0, _backendToUi.backendToUiReportsList)(opensearchResp.reportInstanceList, basePath);
      return response.ok({
        body: {
          data: reportsList
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get reports details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'list', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get single report details by id

  router.get({
    path: `${_common.API_PREFIX}/reports/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'info', 'count');

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: request.params.reportId
      });
      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath);
      return response.ok({
        body: report
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get single report details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'info', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });
}

module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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